
%% 
%% Requires:
%% 
%% FSLNets
%% https://fsl.fmrib.ox.ac.uk/fsl/fslwiki/FSLNets
%% 
%% FACS scatterplot
%% https://uk.mathworks.com/matlabcentral/fileexchange/8430-flow-cytometry-data-reader-and-visualization?focused=6779476&tab=function
%% 
%% FastICA
%% https://research.ics.aalto.fi/ica/fastica/code/dlcode.shtml
%% 

%%% setup matlab and load IDP (etc) workspace
addpath ~steve/NETWORKS/FSLNets ~/matlab/FastICA_25 ~/matlab/FACS
cd /home/fs0/steve/BrainAge
load /vols/Data/ukbiobank/FMRIB/ANALYSIS/workspace10 % clean versions of all IDPs, confounds, nIDPs
FS(FS_T2FLAIR==0,:)=NaN;   % remove FreeSurfer IDPs for subjects that didn't have T1+T2FLAIR

%%% names and types of IDPs
XN=[IDPnames_i; FSnames];
CombinedModalityTypes=[ IDP_modality_types(IDP_modality_types>0)+6 FScategories'+100 ];
CombinedModalityTypes([nets_cellfind(XN,'_SIENAX_',-1) find(CombinedModalityTypes==103)])=1; % FSL and FS global vols
CombinedModalityTypes([nets_cellfind(XN,'_FIRST_',-1) find(CombinedModalityTypes==101 | CombinedModalityTypes==102)])=2; % FSL and FS subcort vols
CombinedModalityTypes(nets_cellfind(XN,'_FAST_',-1))=3; % FSL cortical ROI vols
CombinedModalityTypes(CombinedModalityTypes>103)=CombinedModalityTypes(CombinedModalityTypes>103)-100; % FS cortical misc.

%%% arrange IDPs matrix, age variables, etc.
X = [ ALL_IDPs(:,IDP_modality_types(1:size(ALL_IDPs,2))>0) NODEamps25 NODEamps100 NET25 NET100 FS ];
X=X-nanmedian(X); grot=nanmedian(abs(X)); grot(grot<eps)=nanstd(X(:,grot<eps))/1.48; X=X./grot; X(abs(X)>6)=NaN;
X=nets_inormal(X);  Xgood=sum(isnan(X),2)<50;  X(isnan(X))=randn(size(X(isnan(X))))*0.01;
Y=age; confX=conf(:,(size(confA1,2)+size(confA2,2)+1):end);
Kp=find((sum(isnan([Y confX]),2)==0)&(Xgood==1));  KpN=length(Kp);  Kp=Kp(randperm(KpN));  100*KpN/N  %  87.4% used subjects

%%% load old ICA results (code below) so we make sure we get the same Kp subset of subjects
load('PcaIca-128-62.mat');

%%% use Kp to take subset of subjects with minimal missing data
Y=Y(Kp);  X=nets_demean(X(Kp,:));  confX=nets_normalise(confX(Kp,:));  XconfX=nets_unconfound(X,confX); SEX=sex(Kp);
XconfX = XconfX .* (0.1+abs(corr(Y,XconfX)));  %%% age-weight the IDPs matrix to guide PCA reduction towards age-related IDPs

%%% setup other age variables and higher-power age models
y=  nets_demean(Y);
yy= nets_demean([y  nets_unconfound(nets_demean(y.^2),y)]);   yy = (yy * std(yy(:,1)))   ./ std(yy);
yyy=nets_demean([yy nets_unconfound(nets_demean(y.^3),yy)]);  yyy= (yyy * std(yyy(:,1))) ./ std(yyy);
[~,Yi]=sort(Y);

%%% make sex-split confounds for applying to nIDPS (*not* to brain-age-modes - these have sex deconfounded)
AllConfs=nets_demean(conf(Kp,:));
AllConfs0=AllConfs(:,1:end-1); AllConfs0(SEX==1,:)=NaN; AllConfs0=nets_demean(AllConfs0); AllConfs0(isnan(AllConfs0))=0;
AllConfs1=AllConfs(:,1:end-1); AllConfs1(SEX==0,:)=NaN; AllConfs1=nets_demean(AllConfs1); AllConfs1(isnan(AllConfs1))=0;
AllConfs2=[AllConfs0 AllConfs1 nets_demean(SEX) ones(size(SEX))];
VarsDeconf = nets_unconfound(nets_inormal(vars(Kp,:)),AllConfs2);  % apply sex-split confounds with no within-deconfounding-demeaning

figure('Position',[10 10 600 400]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
hist(sum(~isnan(VarsDeconf))/size(VarsDeconf,1),20);
xlabel('Proportion of subjects with non-missing data'); ylabel('Number of nIDPs');
%%ssprint('BAmissingData.pdf');


%%% ICA on IDPs, including dimensionality estimation (commented out as this section was run once then saved to disk below)
%%% X = pcaU pcaS pcaV'    pcaV' = icaA icaS   X = pcaU pcaS icaA icaS = pcaicaU icaS    pcaicaU = pcaU pcaS icaA
%[icaS,icaA,pcaU,pcaS,pcaV]=ICAdim(XconfX,[60:150]); J=size(icaS,1); JJ=size(icaA,1);
%pcaicaU = pcaU(:,1:JJ) * pcaS(1:JJ,1:JJ) * icaA; 
%grot=sign(corr(Y,pcaicaU)); pcaicaU=pcaicaU.*grot; icaS=icaS.*grot'; icaA=icaA.*grot;
%grot=nets_unconfound(pcaicaU,yyy);  grot=std(pcaicaU)./std(grot);  figure; plot(grot);
%[~,grot]=sort(grot,'descend'); pcaicaU=pcaicaU(:,grot); icaS=icaS(grot,:); icaA=icaA(:,grot);
%grotIDs=ALL_IDPs(:,1); save('PcaIca-128-62.mat','pca*','ica*','Kp','grotIDs');

%%% null test of split-half reproducibility
grotALL=[];
for i=1:100
  grotAGE=nets_demean(randn(size(X,1),1));
  grotX=nets_unconfound(X,confX);  grotX=grotX.*(0.1+abs(corr(grotAGE,grotX)));
  [groticaS,groticaA,grotpcaU,grotpcaS,grotpcaV]=ICAdim(grotX,-128);
  grotpcaicaU = grotpcaU * grotpcaS * groticaA; grot=corr(grotAGE,grotpcaicaU);
  grotALL=[grotALL max(abs(grot))]
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% low-dimensional ICA to help with clustering analysis of high-dimensional modes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% search over all dimensionalities
%grot1=[];
%for Jlow=2:50
%  [icaSlow,icaAlow,pcaUlow,pcaSlow,pcaVlow]=ICAdim(XconfX,-Jlow);
%  pcaicaUlow = pcaUlow(:,1:Jlow) * pcaSlow(1:Jlow,1:Jlow) * icaAlow; 
%  grot=sort(abs(corr(pcaicaUlow,pcaicaU)),'descend');
%  grot1=[grot1; sum(grot(1,:)./grot(2,:)) sum(grot(1,:)-grot(2,:)) ];  
%end
%figure; plot(grot1); % best result (second plot) is J=6
%%% now optimise for J=6
%[icaSlow,icaAlow,pcaUlow,pcaSlow,pcaVlow]=ICAdim(XconfX,6); Jlow=size(icaSlow,1); JJlow=size(icaAlow,1);
%pcaicaUlow = pcaUlow(:,1:JJlow) * pcaSlow(1:JJlow,1:JJlow) * icaAlow; 
%grot=sign(corr(Y,pcaicaUlow)); pcaicaUlow=pcaicaUlow.*grot; icaSlow=icaSlow.*grot'; icaAlow=icaAlow.*grot;
%grot=nets_unconfound(pcaicaUlow,yyy);  grot=std(pcaicaUlow)./std(grot);  figure; plot(grot);
%[~,grot]=sort(grot,'descend'); pcaicaUlow=pcaicaUlow(:,grot); icaSlow=icaSlow(grot,:); icaAlow=icaAlow(:,grot);
%save('PcaIca-128-62-low.mat','pca*low','ica*low');
load('PcaIca-128-62-low.mat');

figure('Position',[10 10 1000 300]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
imagesc(corr(pcaicaUlow,pcaicaU),[-.9 .9]); colorbar;
xlabel('62 population modes'); ylabel('6 population mode-clusters');
%ssprint('BAmodesMultiDimCorr.pdf');

%%% help work out which modes go with which mode-clusters
figure;imagesc(corr(pcaicaUPlow,pcaicaU),[-.9 .9])
grot=abs(corr(pcaicaUlow,pcaicaU));
for i=1:size(grot,2)
  grot2=find(grot(:,i)>.5);
  if size(grot1,1)>0
    disp([i grot2']);
  end
end

%%% version of IDP matrix and ICA subject-modes without sex-deconfounding
XconfXns = nets_unconfound(X,confX(:,1:end-1)) .* (0.1+abs(corr(Y,XconfX)));
pcaicaUns =    nets_normalise((pinv(icaS') * XconfXns')');
pcaicaUnslow = nets_normalise((pinv(icaSlow') * XconfXns')' ./ std(pcaicaUlow));

%%% misc normalisations and partialling
pcaicaUstd=std(pcaicaU);                                 pcaicaUstdlow=std(pcaicaUlow); 
pcaicaU=nets_normalise(pcaicaU);                         pcaicaUlow=nets_normalise(pcaicaUlow);
pcaicaUP=nets_normalise(nets_nodepartial(pcaicaU));      pcaicaUPlow=nets_normalise(nets_nodepartial(pcaicaUlow));
pcaicaUnsP=nets_normalise(nets_nodepartial(pcaicaUns));  pcaicaUnsPlow=nets_normalise(nets_nodepartial(pcaicaUnslow));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% save full workspace to disk
ALL_IDPs_t2starputamen=ALL_IDPs_i_deconf(Kp,170);
clear unicorr* vars_i* FS_i* ALL_IDPs_i* grot* poop*;
%save('workspace-128-62.mat','-v7.3');





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% finally start brain age modelling including deltas

confXY=[yyy nets_demean(conf(Kp,:))]; % full deconfounding model;

%%% switched model
beta3   =pinv(yyy)*pcaicaU;     Yb3=yyy*beta3;       figure;  plot(abs(beta3'));    % beta3 unaffected if use pcaicaU2 (ie with sex in it)
beta3low=pinv(yyy)*pcaicaUlow;  Yb3low=yyy*beta3low; figure;  plot(abs(beta3low'));
delta3ica=(pcaicaU-Yb3)./beta3(1,:);          % crazy delta scaling if age is on X axis
delta3ica=nets_unconfound(delta3ica,confXY);  % more sensible age regression that also makes sure confounds are not induced

Yb3P=yyy*(pinv(yyy)*pcaicaUP); Yb3Plow=yyy*(pinv(yyy)*pcaicaUPlow);  % fitting to partialled modes

figure('Position',[10 10 1000 800]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
  plot(Y(Yi),Yb3(Yi,:)); xlabel('Age'); ylabel('Mean predicted mode curves (62 modes)');
  set(gca,'XLim',[min(Y)-.5 max(Y)+.5],'YLim',[min(Yb3(:))-0.05 max(Yb3(:))+0.05]);  %ssprint('BAmodes62Yb3.pdf');
figure('Position',[10 10 500 400]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
  plot(Y(Yi),Yb3low(Yi,:)); xlabel('Age'); ylabel({'Mean predicted mode curves','(6 mode-clusters)'});
  legend('Mode 1','Mode 2','Mode 3','Mode 4','Mode 5','Mode 6','Location','southeast')
  set(gca,'XLim',[min(Y)-.5 max(Y)+.5],'YLim',[min(Yb3low(:))-0.05 max(Yb3low(:))+0.05]);  %ssprint('BAmodes6Yb3.pdf');

%%% mean curves plus all data points
for I=[1 21 41 62]
  figure('Position',[10 10 700 600]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
  dscatter(Y,pcaicaU(:,I)); hold on; plot(Y(Yi),Yb3(Yi,I),'k','LineWidth',2);
  xlabel('Age'); ylabel(sprintf('Mode subject weights and age-predicted mean curve (mode %d)',I));
  set(gca,'XLim',[min(Y)-.5 max(Y)+.5],'YLim',[min(pcaicaU(:,I))-0.05 max(pcaicaU(:,I))+0.05]);
  %ssprint(sprintf('BAmodes62data_and_fit_%d.pdf',I));
end

%%% error from model 3 in units of pcaicaU (ie before rescaling by beta to get age-delta).
figure('Position',[10 10 1000 260]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
grot=pcaicaU-Yb3; plot(std(Yb3) ./ std(grot),'LineWidth',2); hold on;
grot=pcaicaUlow-Yb3low; plot(std(Yb3low) ./ std(grot),'LineWidth',2);
xlabel('Population mode/mode-cluster number'); ylabel({'Stddev (mean mode curve) /',' Stddev (mode data)'}); set(gca,'XLim',[0 size(pcaicaU,2)+1]);
legend('62 modes','6 mode-clusters');
%ssprint('BAmodesFractionalStddev.pdf');

%%% sex-split aging curves
for JJ=1:4
  if JJ==1
    grotYb3=Yb3; grotU=pcaicaUns; IIII=6;
  elseif JJ==2
    grotYb3=Yb3P; grotU=pcaicaUnsP; IIII=6;
  elseif JJ==3
    grotYb3=Yb3low; grotU=pcaicaUnslow; IIII=1;
  elseif JJ==4
    grotYb3=Yb3Plow; grotU=pcaicaUnsPlow; IIII=1;
  end
  for III=1:IIII
    figure('Position',[10 10 700 800]); set(gcf,'PaperPositionMode','auto','PaperOrientation','portrait');
    for II=1:12
      I=(III-1)*12+II;
      if I<=size(grotYb3,2)
        subplot(4,3,II); hold off;
        plot(Y,grotYb3(:,I),'.','Color',[.7 .7 .7]); hold on; grotZZ=[];
        for SEXi=0:1
          SEXcol=[0 .447 .741]; if SEXi==1, SEXcol=[.85 .325 .098]; end;
          grotZ=[]; grotY=min(Y)+3:max(Y)-3;
          for Yii=grotY
            grot=grotU(SEX==SEXi & Y>Yii-2.5 & Y<Yii+2.5,I);
            grotZ=[grotZ; [ mean(grot) + [ 0 -std(grot)/sqrt(length(grot)) std(grot)/sqrt(length(grot)) ] prctile(grot,[25 75])  ] ];
          end
          plot(grotY,grotZ(:,1:3),'Color',SEXcol,'LineWidth',1);
          plot(grotY,grotZ(:,[4 5]),':','Color',SEXcol);
          grotZZ=[grotZZ;grotZ];
        end
        set(gca,'XLim',[min(Y)-.5 max(Y)+.5],'YLim',[min(grotZZ(:))-0.1 max(grotZZ(:))+0.1],'XTick',[],'YTick',[]);
        sub_pos = get(gca,'position'); sub_pos=sub_pos.*[1 1 1.25 1.3]; set(gca,'position',sub_pos);
        grot=annotation('textbox',[sub_pos(1)+0.005 sub_pos(2)-0.003 0.2 0.2],'String',sprintf('%d',I),'FitBoxToText','on','LineStyle','none');
      end
    end
  ssprint3(sprintf('BAmodesSexCurves_%d_%d.png',JJ,III));
  end
end


%%% main delta fitting, all-in-one
beta1   =pinv(pcaicaU)*y;    Yb1   =pcaicaU.*beta1';       delta2ica   =nets_unconfound(Yb1,confXY);  % no need to subtract y from Yb1, given last step
beta1low=pinv(pcaicaUlow)*y; Yb1low=pcaicaUlow.*beta1low'; delta2icalow=nets_unconfound(Yb1low,confXY);
delta2icaP    = nets_nodepartial(delta2ica);    % partialise deltas
delta2icaPlow = nets_nodepartial(delta2icalow);



%%% stddev(initial fit) (which may or may not include successful age modelling) is close to stddev(delta2)
figure('Position',[10 10 1000 250]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
plot([std(Yb1)' std(delta2ica)' std(delta2icaP)'],'LineWidth',2);
xlabel('Population mode number'); ylabel('Stddev (years)'); set(gca,'XLim',[0 size(pcaicaU,2)+1]);
legend('stddev(predicted age, standard model part 1)','stddev(delta, standard model part2)','stddev(partialled delta)');
%ssprint('BAmodesStddevs1.pdf');
figure('Position',[10 10 500 250]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
plot([std(Yb1low)' std(delta2icalow)' std(delta2icaPlow)'],'LineWidth',2);
xlabel('Population mode-cluster number'); ylabel('Stddev (years)'); set(gca,'XLim',[.9 size(pcaicaUlow,2)+.1],'YLim',[0 3],'XTick',1:6);
legend('stddev(predicted age, standard model part 1)','stddev(delta, standard model part2)','stddev(partialled delta)');
%ssprint('BAmodesStddevs2.pdf');

%%% betas on GLM age prediction
figure('Position',[10 10 500 200]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape'); hold on;
grot=corr(pcaicaU,Y); plot(grot,'LineWidth',2);
xlabel('Population mode number'); ylabel({'correlation (r)','of mode with age'});
set(gca,'XLim',[0 size(pcaicaU,2)+1],'YLim',[0 .55],'YTick',0:0.1:0.5);
%ssprint('BAmodesJustCorrelation.pdf');
figure('Position',[10 10 1000 500]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape'); hold on;
grot3=corr(pcaicaU,Y); plot(grot3,'LineWidth',2);
grot1=beta1; plot(grot1,'LineWidth',2);
grot2=0.01*-log10(t_to_p(ssglmT(y,pcaicaU),size(pcaicaU,1)-size(pcaicaU,2)-1)); plot(grot2,'LineWidth',2);
plot([1 size(pcaicaU,2)],[1 1]*0.01*-log10(0.05/size(pcaicaU,2)),'k:');
plot([1 size(pcaicaU,2)],[0 0],'k');
legend('correlation r','regression beta','regression significance (-log10P/100)','Bonferroni significance level');
xlabel('Population mode number');
grot=[grot1 grot2 grot3];
set(gca,'XLim',[0 size(pcaicaU,2)+1],'YLim',[min(grot(:))-0.1 max(grot(:))+0.1]);
%ssprint('BAmodesRegressionCoeffs1.pdf');
figure('Position',[10 10 500 500]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape'); hold on;
grot3=corr(pcaicaUlow,Y); plot(grot3,'LineWidth',2);
grot1=beta1low; plot(grot1,'LineWidth',2);
grot2=0.01*-log10(t_to_p(ssglmT(y,pcaicaUlow),size(pcaicaUlow,1)-size(pcaicaUlow,2)-1)); grot2(isinf(grot2))=10; plot(grot2,'LineWidth',2);
plot([1 size(pcaicaUlow,2)],[1 1]*0.01*-log10(0.05/size(pcaicaUlow,2)),'k:');
plot([1 size(pcaicaUlow,2)],[0 0],'k');
legend('correlation r','regression beta','regression significance (-log10P/100)','Bonferroni significance level');
xlabel('Population mode-cluster number');
grot=[grot1 grot2 grot3];
set(gca,'XLim',[0.9 size(pcaicaUlow,2)+.1],'YLim',[0 max(grot(grot<10))+1],'XTick',1:6);
%ssprint('BAmodesRegressionCoeffs2.pdf');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% various other brain-age prediction stuff

%%% "traditional" brain-age from all IDPs together
J=55; [delta1,delta2,delta3,delta2q,delta3q] = DeltaCV(X,Y,J,confX);
[grotR,grotP]=corr(delta2q,VarsDeconf,'rows','pairwise'); [mean(abs(delta2q)) prctile(-log10(grotP'),99)] % 2.9160   12.6579

%%% SINGLE-DELTA from ICA model
[delta1icaALL,delta2icaALL,delta3icaALL,delta2qicaALL,delta3qicaALL] = DeltaCV(pcaicaU,Y,0,[]);
[grotRicaALL,grotPicaALL]=corr(delta2qicaALL,VarsDeconf,'rows','pairwise');
     [mean(abs(delta2qicaALL)) prctile(-log10(grotPicaALL'),99)] % 2.9407   10.7477
[delta1icaALLlow,delta2icaALLlow,delta3icaALLlow,delta2qicaALLlow,delta3qicaALLlow] = DeltaCV(pcaicaUlow,Y,0,[]);
[grotRicaALLlow,grotPicaALLlow]=corr(delta2qicaALLlow,VarsDeconf,'rows','pairwise');
     [mean(abs(delta2qicaALLlow)) prctile(-log10(grotPicaALLlow'),99)] % 2.9211    7.4000



%nets_hierarchy(corr(delta2ica),-corr(delta2icaP),[],'',0.6);
%nets_hierarchy(abs(corr(delta2ica)),-corr(delta2icaP),[],'',0.7);
nets_hierarchy(abs(corr(delta2ica)),corr(delta2ica),[],'',0.7);
set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape','Position',[10 10 1500 700],'PaperSize',[20 20]);
%print('-painters','-dpdf','BAmodesNetsHier.pdf');  ! tar cvfz ~/grot BAmodesNetsHier.pdf


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% multiplicative delta

y0=(y-min(y))/(max(y)-min(y));

for I=[2 1] % do 62-modes analysis second, for use in following code
  if I==1
    grotD=delta2ica; 
  else
    grotD=delta2icalow; 
  end;
  clear lambda*;

  for i=1:size(grotD,2)
    [grot,grotstats]=robustfit(nets_demean(y0),nets_demean(log(abs(grotD(:,i)))),'bisquare',1);
    lambdaM(i)=grot(1);  lambda(i)=exp(grot(2))-1;  lambdaP(i)=grotstats.p(2);
  end;

  figure('Position',[10 10 1000 400]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
  plot(lambda,'LineWidth',2); hold on; % lambda is the fractional increase in delta between youngest and oldest subjects
  plot(lambda.*mean(abs(grotD)),'LineWidth',2); hold on; 
  plot(0.05*-log10(lambdaP),'LineWidth',1);
  plot([1 size(grotD,2)],0.05*-log10(0.05/size(grotD,2))*[1 1],'k:');
  plot([1 size(grotD,2)],[0 0],'k');
  legend('Fractional increase in size of delta','Increase in delta (years)','Multiplicative significance -log10(P)/20','Bonferroni threshold');
  xlabel('Population mode number'); set(gca,'XLim',[0.9 size(grotD,2)+.1])
  if I==2, set(gca,'XTick',1:6); xlabel('Population mode-cluster number'); end;
  %ssprint(sprintf('BAmodesMult%d.pdf',I));
end

sum(-log10(lambdaP)>-log10(0.05/size(grotD,2))) % how many pass Bonferroni? 17

%%% example delta scatterplots showing strong lambda
figure('Position',[10 10 800 300]); set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape');
subplot(1,2,1); dscatter(Y,delta2ica(:,4));  set(gca,'XLim',[45 80]); xlabel('True age (y)'); ylabel('Delta, mode 4/62 (y)');
subplot(1,2,2); dscatter(Y,delta2ica(:,11)); set(gca,'XLim',[45 80]); xlabel('True age (y)'); ylabel('Delta, mode 11/62 (y)');
%ssprint('BAmodesMultScatter.pdf');

% show regression fit in log space - used to tune robust regression parameter
i=4; figure; grot=nets_demean(log(abs(delta2ica(:,i)))); groty=nets_demean(y0); dscatter(Y,grot);
hold on; plot([min(Y) max(Y)],lambdaM(i)+[min(groty) max(groty)]*log(lambda(i)+1),'k','LineWidth',2);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% USE GPCs and save for GWAS

for J=1:3
  grotOUTGWAS=[];
  for I=1:2
    if I==1
      grot='GPCs_disco2.txt';
    else
      grot='GPCs_repro2.txt';
    end
    GPCs = nets_normalise(nets_load_match(grot,ALL_IDPs(Kp,1)));  KpG=~isnan(GPCs(:,1));
    grot1=ALL_IDPs(Kp,1); grot1=grot1(KpG);             % GWAS-subjects' IDs
    grot=[delta2ica delta2icaP delta2icalow delta2icaPlow delta1 delta2q mean(delta2ica,2) delta2qicaALL ALL_IDPs_t2starputamen]; 
    grot=grot(KpG,:);  confGWAS=[confXY GPCs];  confGWAS=confGWAS(KpG,:);  grotS=SEX(KpG);
    if J==1, grotS=(grotS>-1); end; % all subjects
    if J==2, grotS=(grotS==0); end; % females only
    if J==3, grotS=(grotS==1); end; % males only
    grot=nets_unconfound( grot(grotS,:) , confGWAS(grotS,:) ); grot1=grot1(grotS);
    grotOUTGWAS=[grotOUTGWAS;  grot1 grot]; [ size(grotOUTGWAS) size(grot) ]
  end
  grot=sprintf('GWAS_2019_09_02_%d.mat',J)
  save(grot,'grotOUTGWAS');
end

%%%%%%%%%%%%%%% NOW DO sex-separated gwases

% what is second-line threshold level for Manhattan?
-log10( 10^-7.5 / size(delta2ica,2) )                          % 9.29
-log10( 10^-7.5 / (size(delta2ica,2)+size(delta2icalow,2)) )   % 9.33

%%%%%%%%%%%%%%%%%%%%%%%%% then run GWAS.

%%% load GWAS outputs back in
gwasP=load('GWAS_2019_09_02_1/p.txt');   % gwasB=load('GWAS_2019_09_02/b.txt'); 
gwasSTATS=readtable('GWAS_2019_09_02_1/stats.txt', 'Delimiter', 'space');
groti=find(strcmp(gwasSTATS.chr,'0X')==1); for i=groti', gwasSTATS.chr{i}='23'; end;  gwasSTATS.chr=str2double(gwasSTATS.chr);

% get subsets of gwas logP and control MAF at 1%/99%
grot=gwasP; grot(gwasSTATS.af<0.01 | gwasSTATS.af>0.99,:)=1e-10;
gwasPP=     grot(:,1:size(icaS,1));
gwasPPp=    grot(:,size(icaS,1)+1:1:size(icaS,1)*2);
gwasPPlow=  grot(:,size(icaS,1)*2+1:size(icaS,1)*2+size(icaSlow,1));
gwasPPplow= grot(:,size(icaS,1)*2+size(icaSlow,1)+1:size(icaS,1)*2+size(icaSlow,1)*2);
gwasPPmisc= grot(:,size(icaS,1)*2+size(icaSlow,1)*2+1:end);
clear grot*;

grotGWAS=gwasPP;              GWASplot;  ssprint2('BAmodesGWAS_62.png');       close;
grotGWAS=gwasPPp;             GWASplot;  ssprint2('BAmodesGWAS_62p.png');      close;
grotGWAS=gwasPPlow;           GWASplot;  ssprint2('BAmodesGWAS_6.png');        close;
grotGWAS=gwasPPplow;          GWASplot;  ssprint2('BAmodesGWAS_6p.png');       close;
grotGWAS=gwasPPmisc(:,[4 2]); GWASplot;  ssprint2('BAmodesGWAS_AllInOne.png'); close;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% combine plotting with Elliott ALL-IDPs-GWAS results

gwasOLD=load('/vols/Data/ukbiobank/FMRIB/GENETICS/15k/zscores.mat');
[gwasOLD.Z,gwasOLD.Zmaxn]=max(abs(gwasOLD.Z)); %% collapse across IDPs - might not want to do that for fuller investigations
gwasOLD.P=-log10(t_to_p(gwasOLD.Z,1e10));

%%%% get vector of chr starting positions
gwasPOS=[0]; % start position for each chromosome
for I=1:max(gwasSTATS{:,1})
  grot1=gwasSTATS.pos(find(gwasSTATS{:,1}==I));
  grot2=gwasOLD.columns.pos(find(gwasOLD.columns{:,1}==I));
  if size(grot2,1)==0, grot2=grot1; end; % for ChrX, missing in OLD GWAS
  grot=[min(min(grot1),min(grot2)) max(max(grot1),max(grot2))]
  gwasPOS=[gwasPOS; gwasPOS(end)+grot(2)+1e7];
end

grotGWAS=gwasPP; GWASplotWithOld
savefig(fig,'GWASplotWithOld.fig','compact');

%%% GWAS with OLD (univariate GWAS) in grey crosses, 62-modes in dark green, and all-in-one deltas in other colours
grotGWAS=gwasPP; grotGWASb=gwasPPmisc(:,[4 2]); GWASplotWithOldAndAllinone
savefig(fig,'GWASplotWithOldAndAllinone.fig','compact');




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% correlations with nIDPs
grotd=delta2ica;               grot=VarsDeconf;           [VarsCorrR,VarsCorrP]=    corr(grotd,grot,'rows','pairwise'); VarsCorrN=  sum(~isnan(grot));
grotd=delta2ica(SEX==0,:);     grot=VarsDeconf(SEX==0,:); [VarsCorrRF,VarsCorrPF]=  corr(grotd,grot,'rows','pairwise'); VarsCorrNF= sum(~isnan(grot));
grotd=delta2ica(SEX==1,:);     grot=VarsDeconf(SEX==1,:); [VarsCorrRM,VarsCorrPM]=  corr(grotd,grot,'rows','pairwise'); VarsCorrNM= sum(~isnan(grot));
grotd=delta2icaP;              grot=VarsDeconf;           [VarsCorrRp,VarsCorrPp]=  corr(grotd,grot,'rows','pairwise');
grotd=delta2icaP(SEX==0,:);    grot=VarsDeconf(SEX==0,:); [VarsCorrRFp,VarsCorrPFp]=corr(grotd,grot,'rows','pairwise');
grotd=delta2icaP(SEX==1,:);    grot=VarsDeconf(SEX==1,:); [VarsCorrRMp,VarsCorrPMp]=corr(grotd,grot,'rows','pairwise');
grotd=delta2icalow;            grot=VarsDeconf;           [VarsCorrRlow,VarsCorrPlow]=    corr(grotd,grot,'rows','pairwise');
grotd=delta2icalow(SEX==0,:);  grot=VarsDeconf(SEX==0,:); [VarsCorrRFlow,VarsCorrPFlow]=  corr(grotd,grot,'rows','pairwise');
grotd=delta2icalow(SEX==1,:);  grot=VarsDeconf(SEX==1,:); [VarsCorrRMlow,VarsCorrPMlow]=  corr(grotd,grot,'rows','pairwise');
grotd=delta2icaPlow;           grot=VarsDeconf;           [VarsCorrRplow,VarsCorrPplow]=  corr(grotd,grot,'rows','pairwise');
grotd=delta2icaPlow(SEX==0,:); grot=VarsDeconf(SEX==0,:); [VarsCorrRFplow,VarsCorrPFplow]=corr(grotd,grot,'rows','pairwise');
grotd=delta2icaPlow(SEX==1,:); grot=VarsDeconf(SEX==1,:); [VarsCorrRMplow,VarsCorrPMplow]=corr(grotd,grot,'rows','pairwise');


for JJ=1:4
  if JJ==1
    grotP=VarsCorrP; grotPF=VarsCorrPF; grotPM=VarsCorrPM; grotR=VarsCorrR; grotRF=VarsCorrRF; grotRM=VarsCorrRM;
  elseif JJ==2
    grotP=VarsCorrPp; grotPF=VarsCorrPFp; grotPM=VarsCorrPMp; grotR=VarsCorrRp; grotRF=VarsCorrRFp; grotRM=VarsCorrRMp;
  elseif JJ==3
    grotP=VarsCorrPlow; grotPF=VarsCorrPFlow; grotPM=VarsCorrPMlow; grotR=VarsCorrRlow; grotRF=VarsCorrRFlow; grotRM=VarsCorrRMlow;
  elseif JJ==4
    grotP=VarsCorrPplow; grotPF=VarsCorrPFplow; grotPM=VarsCorrPMplow; grotR=VarsCorrRplow; grotRF=VarsCorrRFplow; grotRM=VarsCorrRMplow;
  end
  grotN=VarsCorrN; grotNF=VarsCorrNF; grotNM=VarsCorrNM;
  grotF=sprintf('BrainAgeModesTable%d.xlsx',JJ);  system(sprintf('/bin/mv %s %sOLD',grotF,grotF));
  for J=1:size(grotP,1)
    [~,grotI]=sort(-log10(grotP(J,:)'),'descend'); clear grotA*;
    ii=1; for i= 1:length(grotI)
      I=grotI(i); grotp=[round(-log10(grotP(J,I))*10)/10 round(-log10(grotPF(J,I))*10)/10 round(-log10(grotPM(J,I))*10)/10];
      if max(grotp)>5
        grotA1(ii)=grotp(1);                       grotA5(ii)=grotp(2);                        grotA9(ii)=grotp(3);
        grotA2(ii)=round(grotR(J,I)*100)/100;      grotA6(ii)=round(grotRF(J,I)*100)/100;      grotA10(ii)=round(grotRM(J,I)*100)/100;
        grotA3(ii)=round(abs(grotR(J,I))*100)/100; grotA7(ii)=round(abs(grotRF(J,I))*100)/100; grotA11(ii)=round(abs(grotRM(J,I))*100)/100;
        grotA4(ii)=abs(grotN(I));                  grotA8(ii)=abs(grotNF(I));                  grotA12(ii)=abs(grotNM(I));
        %grotA=varsVARS{varskeep(I)}; grotA=grotA(1:strfind(varsVARS{varskeep(I)},'-')-1);
        %grotA13{ii}=sprintf('=HYPERLINK("https://biobank.ctsu.ox.ac.uk/showcase/field.cgi?id=%s","%s")',grotA,varsVARS{varskeep(I)});
        grotA13{ii}=varsVARS{varskeep(I)};
        grotA14{ii}=varsHeader{varskeep(I)};
        ii=ii+1;
      end
    end
    if exist('grotA1')
      grot=table(grotA1',grotA2',grotA3',grotA4',grotA5',grotA6',grotA7',grotA8',grotA9',grotA10',grotA11',grotA12',grotA13',grotA14',...
        'VariableNames',{'log10P','R','absR','N','log10P_F','R_F','absR_F','N_F','log10P_M','R_M','absR_M','N_M','VariableID','VariableName'});
      writetable(grot,grotF,'Sheet',J);
    end
  end
end
! tar cvfz ~/grot BrainAgeModesTable?.xlsx

%%% now outputs for IDP weights
for JJ=1:2
  if JJ==1
    grotS=icaS;
  elseif JJ==2
    grotS=icaSlow;
  end
  grotF=sprintf('BrainAgeModesTableIDPs%d.xlsx',JJ);  system(sprintf('/bin/mv %s %sOLD',grotF,grotF));
  for J=1:size(grotS,1)
    [grotY,grotI]=sort(abs(grotS(J,:)),'descend'); clear grotA*;  ii=1;
    for i=1:length(grotI)
      if grotY(i)>5
        I=grotI(i); grotA1(ii)=round(grotS(J,I)*10)/10; grotA2i(ii)=I; grotA2{ii}=XN{I}; grotA3(ii)=CombinedModalityTypes(I); ii=ii+1;
      end
    end
    if exist('grotA1')
      grot=table(grotA1',grotA2i',grotA2',grotA3','VariableNames',{'IDPweight','IDPIndex','IDPName','Modality'});
      writetable(grot,grotF,'Sheet',J);
    end
  end
end
! tar cvfz ~/grot BrainAgeModesTableIDPs?.xlsx

ssplot(icaS'); set(gca,'YLim',[-4 64],'YTick',5:5:62);
set(gcf,'Position',[10 10 1600 900],'PaperPositionMode','auto','PaperOrientation','landscape','PaperPosition',[0 0 10 5]);
%ssprint('BAmodesSummaryIDPs62.pdf');
ssplot(icaSlow'); set(gca,'YLim',[-1 7],'YTick',1:6);
set(gcf,'Position',[10 10 1600 300],'PaperPositionMode','auto','PaperOrientation','landscape','PaperPosition',[0 0 10 5]);
%ssprint('BAmodesSummaryIDPs6.pdf');

grotF='BAmodesIDPlist.xlsx';  system(sprintf('/bin/mv %s %sOLD',grotF,grotF)); clear grotA*;
for J=1:size(XN,1)
  grotA1(J)=J; grotA2{J}=XN{J}; grotA3(J)=CombinedModalityTypes(J);
end
grot=table(grotA1',grotA2',grotA3','VariableNames',{'IDPindex','IDPName','Modality'});
writetable(grot,grotF,'Sheet',1);   ! tar cvfz ~/grot BAmodesIDPlist.xlsx

%%% save workspace out for correlation against subject-level maps
clear vox*;
vox.IDs=ALL_IDPs(Kp,1);
vox.delta2ica=delta2ica;
vox.delta2icaP=delta2icaP;
vox.delta2icalow=delta2icalow;
vox.delta2icaPlow=delta2icaPlow;
%vox.delta2q=delta2q;
%vox.delta2qicaALL=delta2qicaALL;
vox.conf=confXY;
vox.sex=SEX;
save('vox','vox');
clear vox*;

grot=-log10(0.05/(2e6+6*1.4e5))  %% what is the log10p Bonferroni threshold for all the voxelwise tests?  7.75
VarsCorrR(find( -log10(VarsCorrP(:)) > grot*0.999 & -log10(VarsCorrP(:)) < grot*1.001))  %% corresponding r?  r=0.1

%%% make netmat/nodeamp pics
icaTHRESH=5;
groticaS=icaSlow; J=5;
groticaS=icaS; J=17; % 17 31 33 45 52 58

for J=1:size(groticaS,1)
grot=groticaS(J,CombinedModalityTypes==12); grot=grot(:,1:21); % NODEAMP25
  if max(abs(grot))>icaTHRESH
  clear fakets; fakets.DD=load('~/biobank/FMRIB/IMAGING/data3/group/rfMRI_GoodComponents_d25_v1.txt'); fakets.Nnodes=length(fakets.DD);
  nets_nodeweightpics(fakets,'~/biobank/FMRIB/IMAGING/data3/group/groupICA_d25.ica/melodic_IC.sum',grot,16,2);
  set(gcf,'PaperSize',[20 20]); ssprint(sprintf('BAmodesRFMRIpics_%d_%d_25.pdf',size(groticaS,1),J)); end;
grot=groticaS(J,CombinedModalityTypes==12); grot=grot(:,22:end); % NODEAMP100
  if max(abs(grot))>icaTHRESH
  clear fakets; fakets.DD=load('~/biobank/FMRIB/IMAGING/data3/group/rfMRI_GoodComponents_d100_v1.txt'); fakets.Nnodes=length(fakets.DD);
  nets_nodeweightpics(fakets,'~/biobank/FMRIB/IMAGING/data3/group/groupICA_d100.ica/melodic_IC.sum',grot,16,2);
  set(gcf,'PaperSize',[20 20]); ssprint(sprintf('BAmodesRFMRIpics_%d_%d_100.pdf',size(groticaS,1),J)); end;
grot=groticaS(J,CombinedModalityTypes==13); grot=grot(:,1:210); % NET25
  if max(abs(grot))>icaTHRESH
  clear fakets; fakets.DD=load('~/biobank/FMRIB/IMAGING/data3/group/rfMRI_GoodComponents_d25_v1.txt'); fakets.Nnodes=length(fakets.DD);
  grot1=zeros(fakets.Nnodes);  grot1(triu(ones(fakets.Nnodes),1)>0)=grot';
  grotM=zeros(fakets.Nnodes);  grotM(triu(ones(fakets.Nnodes),1)>0)=nanmean(NET25); grotM=grotM+grotM';
  nets_edgepics(fakets,'~/biobank/FMRIB/IMAGING/data3/group/groupICA_d25.ica/melodic_IC.sum',grotM,grot1,24,2); 
  set(gcf,'PaperSize',[20 20],'PaperPositionMode','auto','Position',[10 10 1800 900]);
  ssprint(sprintf('BAmodesRFMRIedgepics_%d_%d_25.pdf',size(groticaS,1),J)); end;
grot=groticaS(J,CombinedModalityTypes==13); grot=grot(:,211:end); % NET100
  if max(abs(grot))>icaTHRESH
  clear fakets; fakets.DD=load('~/biobank/FMRIB/IMAGING/data3/group/rfMRI_GoodComponents_d100_v1.txt'); fakets.Nnodes=length(fakets.DD);
  grot1=zeros(fakets.Nnodes);  grot1(triu(ones(fakets.Nnodes),1)>0)=grot';
  grotM=zeros(fakets.Nnodes);  grotM(triu(ones(fakets.Nnodes),1)>0)=nanmean(NET100); grotM=grotM+grotM';
  nets_edgepics(fakets,'~/biobank/FMRIB/IMAGING/data3/group/groupICA_d100.ica/melodic_IC.sum',grotM,grot1,24,2); 
  set(gcf,'PaperSize',[20 20],'PaperPositionMode','auto','Position',[10 10 1800 900]); 
  ssprint(sprintf('BAmodesRFMRIedgepics_%d_%d_100.pdf',size(groticaS,1),J)); end;
end

! tar cvfz ~/grot BAmodesRFMRI*




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% make a plot that summarises which IDP cateogries, nIDP categories
%%% and genetic involvement are in each mode, as a decreasing function
%%% of age-prediction extent

NM=size(icaS,1); grotPAD=zeros(NM,2);
poop0=std(delta2ica)'; poop0=[grotPAD repmat(poop0/(1.5*max(poop0)),1,5) grotPAD];  % delta stddev

poop1=grotPAD; grotm=max(abs(icaS(:)));
for J=1:max(CombinedModalityTypes)
  grot=icaS(:,CombinedModalityTypes==J)'; %ssplot(grot,-1);
  if (size(grot,1)>1)
    [u,s,v]=nets_svds(nets_demean(grot),0); ss=diag(s); ssm=ss(ceil(end*0.75)); ss=min(size(s,1)-1,max(5,1+sum(((ss-ssm)/(ss(1)-ssm))>0.1)));
    [grotS,grotA,~]=fastica(grot','approach','symm','g','tanh','epsilon',1e-13,'maxNumIterations',3000,'lastEig',ss);
    v=grotA .* max(abs(grotS')); v=v.*sign(mean(v));  poop=max(abs(v));
    if max(poop)>grotm*0.2; % or 0.4
      v=v(:,poop>grotm*0.2);
    else
      v=v(:,find(poop==max(poop)));
    end
    [~,poop]=sort(max(abs(v)),'descend'); v=v(:,poop);
  else
    v=grot';
  end
  poop1=[poop1 kron(v,ones(1,2)) grotPAD];
end
poop1=poop1/max(abs(poop1(:)));

poop2=grotPAD; grotP=-log10(VarsCorrP); grotP(isnan(grotP) | isinf(grotP))=0;
for J=vt_use
  grot=grotP(:,varskeepVT==J)'; % ssplot(grot,-1);
  [u,s,v]=nets_svds(nets_demean(grot),0); ss=diag(s); ssm=ss(ceil(end*0.75)); ss=min(size(s,1)-1,max(5,1+sum(((ss-ssm)/(ss(1)-ssm))>0.05)));
  [grotS,grotA,~]=fastica(grot','approach','symm','g','tanh','epsilon',1e-13,'maxNumIterations',3000,'lastEig',ss);
  v=grotA .* max(abs(grotS')); v=v.*sign(mean(v)); poop=max(abs(v));
  if max(poop)>4  % or 12
    v=v(:,poop>4);
  else
    v=v(:,find(poop==max(poop)));
  end
  [~,poop]=sort(max(abs(v)),'descend'); v=v(:,poop);
  poop2=[poop2 kron(v,ones(1,2)) grotPAD];
end
poop2=poop2/max(abs(poop2(:))); poop2=sqrt(abs(poop2)).*sign(poop2);

poop3=grotPAD; grotMinThr=3; grotMaxThr=9.3-grotMinThr; % with higher MinThr, there were not enough background SNPs for ICA to work well
for J=unique(gwasSTATS.chr)'
  grot=gwasPP(gwasSTATS.chr==J,:); grot=grot-grotMinThr; grot(grot<0)=0; grot=grot(max(grot')>1.5e-10,:); % ssplot(grot,-1);
  [u,s,v]=nets_svds(nets_demean(grot),0); ss=diag(s); ssm=ss(ceil(end*0.75)); ss=min(size(s,1)-1,max(5,1+sum(((ss-ssm)/(ss(1)-ssm))>0.05)));
  [grotS,grotA,~]=fastica(grot','approach','symm','g','tanh','epsilon',1e-13,'maxNumIterations',3000,'lastEig',ss);
  v=grotA .* max(abs(grotS')); v=v.*sign(mean(v)); poop=max(abs(v));                   % scale mixing matrix according to max of grotS (ICA sources)
  if max(poop)>4.5   % or grotMaxThr                                          % do we have modes above threshold?
    v=v(:,poop>4.5);
  else
    v=v(:,find(poop==max(poop)));                                  % if not, show the one stongest anyway
  end
  [~,poop]=sort(max(abs(v)),'descend'); v=v(:,poop);
  poop3=[poop3 kron(v,ones(1,2)) grotPAD];
end
poop3=poop3/max(abs(poop3(:))); poop3=sqrt(abs(poop3)).*sign(poop3);

poop=[poop0 grotPAD grotPAD poop1 grotPAD grotPAD poop2 grotPAD grotPAD poop3];  poop=min(max(poop*1.5,-.95),.95);
poop=kron(poop,ones(10,1)); poop=[poop;zeros(1,size(poop,2))]; poop(1:10:end,:)=0; poop(poop==0)=Inf;
figure('Position',[10 10 1600 900]);
colormap('jet'); grotc=colormap;  grotc(end,:)=[.7 .7 .7];  colormap(grotc);  imagesc(poop,[-1 1]); axis off; daspect('auto');
set(gcf,'PaperPositionMode','auto','PaperOrientation','landscape','PaperPosition',[0 0 10 5]);
%ssprint('BAmodesSummary.pdf');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

